% Matlab code for the simulation of the minimal physical model presented in "Persistent cell migration emerges from a coupling between protrusion dynamics and polarized trafficking by Vaidiulyte et al.
% The code should be run section by section, figures are shown on the way to illustrate the steps
% Written by M. Coppey, 03-25-2021
%% parameters of the model
clear all
tt = 1000; % total duration of the simulation
L = 101; % number of markers along the cell contour
pp = 0.25; % probability of protrusion per unit of time
simgaNG = 0.8; % spatial extent of the protrusion bias toward the polarity axis
speedNG = 0.5; % speed of the polarity axis reorientation
feed1 = 1; % feedback polarity axis --> protrusion on =1, off =0
feed2 = 1; % feedback protrusion --> polarity axis on =1, off =3
pG = 0.8; % probability to make a protrusion in front of the polarity axis
NGforce = 0.8; % bias of the polarity axis towards protrusions
%% Step 1: synthetic morphodynamic map for a single protrusion
% Transfer matrix
Ttot = 100;
% create transfer matrix
x = -50:50;
tt = 0:Ttot;
H = zeros(Ttot+1,101); % H transfer function
c = 0.5; % speed
f1 = 0.1; % fade factor 1
f2 = 0.1; % fade factor 2
f3 = 0.2; % fade factor 3
figure(1);hold on
for t=0:Ttot
    % initial lateral inhibition:
    gauss1 = -0.01*exp(-f1*t);
    % traveling positive waves:
    gauss2 = (0.5.*normpdf(x,c.*t,5)+0.5.*normpdf(x,-c.*t,5))*exp(-f2*t);
    % central long lasting inhibition
    gauss3 = -(0.1.*normpdf(x,0,3).*normpdf(t,35,25).*sqrt(2*pi).*20);
    % central very peaked positve function
    gauss4 = 0.1.*normpdf(x,0,1).*normpdf(t,10,5).*sqrt(2*pi);
    h0 = gauss2+gauss1+gauss3+gauss4;
    H(t+1,:)=h0;
end
% ensure zero mean
int = sum(sum(H));
H = H-int./numel(H);
maxt=max(max(H));
Hf = flipud(H);
% convolve with signal
signal= zeros(Ttot+1,101); % H transfer function
f1 = 0.1; % time scale to appear
f2 = 1; % duration
f3 = 1; % fading time scale
for t=0:Ttot
    if t<f2
        signal(t+1,:)= (1-exp(-f1.*t)).*normpdf(x,0,5);
    else
        signal(t+1,:)= exp(-f3.*(t-f2)).* (1-exp(-f1.*t)).*normpdf(x,0,5);
    end
end
% ensure 1 mean
int = sum(sum(signal));
signal = signal-int./numel(signal);
M = conv2(signal,H);
M = M(1:Ttot+1,50:150);
H = M;
int = sum(sum(H));
H = H-int./numel(H);
figure(1)
imshow(flipud(H),[],'initialmagnification',500)
colormap('jet')
%% run the simulation
%% Step 2: simulate the whole morphodynamic map 
tt = 1000; % total duration of the simulation
L = 101; % number of markers along the contour
Mk = zeros(L,tt); % Morphodynamic map
pos = [];
np = 0; % number of protrusions
NG = zeros(1,tt); % positions of the polarity axis over time
NG(1) = ceil(L*rand); % initial position of the polarity axis, randomy chosen
NGprev = NG(1);
% generate NG axis natural evolution of positioning
va = smooth(10*L*rand(tt+1,1),100,'rloess');
NGn = diff(va);
%------------------------------ main loop over time
for t=1:tt
    NG(t) = NGprev;
    if rand<pp % compute if a new protrusion is made
        if feed1==1
            if rand>pG
                posloc = ceil(L*rand); % random
            else
                posloc = round(normrnd(NG(t),simgaNG)); % in front of the polarity axis
            end
            if posloc>L
                posloc=posloc-L;
            elseif posloc<1
                posloc = L+posloc;
            end
            pos = [pos;posloc,t]; % feedback on
            np =np+1;
        end
    end
    % compute the morphodyanmic map given protrusions
    for k=1:np
        pro = pos(k,:); % current protrusion
        if t-pro(2)<Ttot+1 %time of the protrusion
            locM = H(t-pro(2)+1,:);
            locM = circshift(locM,50+pro(1));
            Mk(:,t) = Mk(:,t)+locM';
        end
    end
    % update polarity axis positioning
    if feed2==1
        dF = 0;
        for k=1:L
            if (abs(k-NG(t))>=L-abs(NG(t)-k))
                distpt = L-abs(NG(t)-k);
                if Mk(k,t)>=0
                    dF(k) = -sign(k-NG(t))*Mk(k,t);
                end
            else
                distpt = abs(k-NG(t));
                if Mk(k,t)>=0
                    dF(k) = sign(k-NG(t))*Mk(k,t);
                end
            end
        end
        NG(t) = NG(t)+speedNG*(NGforce*sum(dF)+(1-NGforce)*NGn(t));
        if NG(t)>L
            NG(t)=NG(t)-L;
        elseif NG(t)<1
            NG(t) = L+NG(t);
        end
        NGprev = NG(t);
    end
end
%------------------------------  end of main loop over time
figure(2)
subplot(2,2,[1,2])
imshow(Mk,[],'initialmagnification',200)
colormap('jet')
hold on
plot((1:tt),NG,'.k')
%% Step 3: make a trajectory from the morphomap
angles = [0:2*pi/100:2*pi];
dx=cos(angles);
dy=sin(angles);
posx = 0;
posy = 0;
for i=1:tt
    posx(i+1) = posx(i)+sum(dx'.*Mk(:,i));
    posy(i+1) = posy(i)+sum(dy'.*Mk(:,i));
end
Px(:)=posx;
Py(:)=posy;

figure(2)
subplot(2,2,3)
set(gca,'FontSize',12,'fontweight','bold')
set(gcf,'color','w');
x0=100;
y0=100;
% width=200;
% height=200;
set(gcf,'units','points','position',[x0,y0,width,height])
xlim([-200 200])
ylim([-200 200])
hold on
plot(Py(:),Px(:),'-')
axis('square')
box on
%% Step 4: calculate autocorrelation of direction
for n=1:tt/20
    vac(n) = 0;
    count = 0;
    for i=1:tt-n
        normloc = (((Px(i)-Px(i+1))^2+(Py(i)-Py(i+1))^2)*((Px(i+n)-Px(i+n+1))^2+(Py(i+n)-Py(i+n+1))^2));
        if normloc~=0
            vac(n)=vac(n)+((Px(i)-Px(i+1))*(Px(i+n)-Px(i+n+1))+(Py(i)-Py(i+1))*(Py(i+n)-Py(i+n+1)))./sqrt(normloc);
            count = count+1;
        end
    end
    vac(n)=vac(n)/count;
end
% compute the mean correlation

figure(2)
subplot(2,2,4)

box on
xx=1:50;
plot(xx,vac)
set(gca,'FontSize',12,'fontweight','bold')
set(gcf,'color','w');